// ==UserScript==
// @name         5ch サムネイル表示（動画専用）
// @namespace    https://example.com/
// @version      2.1.1
// @description  5ch投稿の動画（YouTube、mp4/webm、TikTokCDNなど）のみサムネイル表示し、クリックでポップアップ再生。画像は除外。
// @match        *://*.5ch.net/test/read.cgi/*
// @grant        none
// ==/UserScript==

(function () {
    'use strict';

    const jumpPrefix = 'http://jump.5ch.net/?';
    const PROCESSED_ATTR = 'data-thumbnail-processed';
    const CHECK_INTERVAL = 3000;

    const imageExtensions = ['.jpg', '.jpeg', '.png', '.gif', '.bmp', '.webp'];
    const videoExtensions = ['.mp4', '.webm', '.ogg'];

    function extractYouTubeId(url) {
        let id = null;
        let startSeconds = null;

        let m = url.match(/youtu\.be\/([^?&]+)/);
        if (m) id = m[1];

        if (!id) {
            m = url.match(/[?&]v=([^?&]+)/);
            if (m) id = m[1];
        }

        if (!id) {
            m = url.match(/youtube\.com\/live\/([^?&/]+)/);
            if (m) id = m[1];
        }

        if (!id) {
            m = url.match(/youtube\.com\/shorts\/([^?&/]+)/);
            if (m) id = m[1];
        }

        const tMatch = url.match(/[?&]t=([0-9hms]+)/);
        if (tMatch) {
            const tValue = tMatch[1];
            if (/^\d+$/.test(tValue)) {
                startSeconds = parseInt(tValue, 10);
            } else {
                const timeParts = tValue.match(/(?:(\d+)h)?(?:(\d+)m)?(?:(\d+)s)?/);
                if (timeParts) {
                    const [, h, m, s] = timeParts.map(x => parseInt(x || '0', 10));
                    startSeconds = h * 3600 + m * 60 + s;
                }
            }
        }

        return id ? { id, startSeconds } : null;
    }

    function extractUrls(post) {
        const anchors = post.querySelectorAll('a[href]');
        const urls = [];
        anchors.forEach(a => {
            const href = a.getAttribute('href');
            if (!href) return;
            let url = href.startsWith(jumpPrefix) ? href.substring(jumpPrefix.length) : href;
            urls.push(url);
        });
        return [...new Set(urls)];
    }

    function isVideoUrl(url) {
        const lowerUrl = url.toLowerCase();
        if (videoExtensions.some(ext => lowerUrl.endsWith(ext))) return true;
        if (lowerUrl.includes('v16m-default.tiktokcdn.com')) return true;
        return false;
    }

    function addThumbnails(post) {
        if (post.hasAttribute(PROCESSED_ATTR)) return;

        let thumbWrapper = post.querySelector('div[data-thumbnail-wrapper]');
        if (!thumbWrapper) {
            thumbWrapper = document.createElement('div');
            thumbWrapper.setAttribute('data-thumbnail-wrapper', 'true');
            post.appendChild(thumbWrapper);
        }

        const urls = extractUrls(post);

        urls.forEach(rawUrl => {
            let url = rawUrl.startsWith(jumpPrefix) ? rawUrl.substring(jumpPrefix.length) : rawUrl;
            const lowerUrl = url.toLowerCase();

            if (
                lowerUrl.includes('twitter.com') ||
                lowerUrl.includes('x.com') ||
                lowerUrl.includes('t.co') ||
                lowerUrl.includes('pbs.twimg.com') ||
                lowerUrl.includes('video.twimg.com')
            ) return;

            const exists = [...thumbWrapper.querySelectorAll('video[data-thumbnail-url], img[data-thumbnail-url]')]
                .some(el => el.getAttribute('data-thumbnail-url') === url);
            if (exists) return;

            const ytInfo = extractYouTubeId(url);
            if (ytInfo) {
                const { id, startSeconds } = ytInfo;
                const thumb = document.createElement('img');
                thumb.src = `https://img.youtube.com/vi/${id}/hqdefault.jpg`;
                thumb.title = 'クリックでYouTube動画ポップアップ';
                thumb.style.maxWidth = '120px';
                thumb.style.maxHeight = '90px';
                thumb.style.margin = '12px';
                thumb.style.cursor = 'pointer';
                thumb.style.border = '1px solid #ccc';
                thumb.style.borderRadius = '4px';
                thumb.setAttribute('data-thumbnail-url', url);
                thumb.addEventListener('click', () => openYouTubePopup(id, startSeconds));
                thumbWrapper.appendChild(thumb);
                return;
            }

            if (isVideoUrl(url)) {
                const videoThumb = document.createElement('video');
                videoThumb.src = url;
                videoThumb.title = 'クリックで動画再生ポップアップ';
                videoThumb.style.maxWidth = '120px';
                videoThumb.style.maxHeight = '90px';
                videoThumb.style.margin = '4px';
                videoThumb.style.cursor = 'pointer';
                videoThumb.style.border = '1px solid #ccc';
                videoThumb.style.borderRadius = '4px';
                videoThumb.muted = true;
                videoThumb.loop = true;
                videoThumb.autoplay = true;
                videoThumb.setAttribute('data-thumbnail-url', url);
                videoThumb.addEventListener('click', () => openVideoPopup(url));
                thumbWrapper.appendChild(videoThumb);
            }
        });

        post.setAttribute(PROCESSED_ATTR, 'true');
    }

    const observer = new IntersectionObserver(entries => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                addThumbnails(entry.target);
                observer.unobserve(entry.target);
            }
        });
    }, { threshold: 0.1 });

    function observePosts() {
        const targets = document.querySelectorAll('.post-content, .popup-box');
        targets.forEach(post => {
            if (!post.hasAttribute(PROCESSED_ATTR)) {
                observer.observe(post);
            }
        });
    }

    const mutationObserver = new MutationObserver(mutations => {
        for (const mutation of mutations) {
            for (const node of mutation.addedNodes) {
                if (node.nodeType !== 1) continue;

                if (node.matches('.post-content, .popup-box')) {
                    observer.observe(node);
                }

                const matchingNodes = node.querySelectorAll?.('.post-content, .popup-box');
                if (matchingNodes?.length) {
                    matchingNodes.forEach(el => observer.observe(el));
                }
            }
        }
    });

    mutationObserver.observe(document.body, {
        childList: true,
        subtree: true
    });

    function createOverlay() {
        const overlay = document.createElement('div');
        overlay.style.position = 'fixed';
        overlay.style.top = 0;
        overlay.style.left = 0;
        overlay.style.width = '100vw';
        overlay.style.height = '100vh';
        overlay.style.backgroundColor = 'rgba(0,0,0,0.8)';
        overlay.style.zIndex = 999999;
        overlay.style.display = 'flex';
        overlay.style.justifyContent = 'center';
        overlay.style.alignItems = 'center';
        overlay.style.cursor = 'pointer';
        overlay.style.overflow = 'auto';
        return overlay;
    }

    function openVideoPopup(url) {
        const overlay = createOverlay();

        const video = document.createElement('video');
        video.src = url;
        video.controls = true;
        video.autoplay = true;
        video.style.maxWidth = '90vw';
        video.style.maxHeight = '90vh';
        video.style.borderRadius = '6px';
        video.style.boxShadow = '0 0 20px rgba(0,0,0,0.8)';
        video.style.display = 'block';
        video.style.margin = 'auto';

        overlay.appendChild(video);
        document.body.appendChild(overlay);

        overlay.addEventListener('click', () => {
            video.pause();
            document.body.removeChild(overlay);
        });
    }

    function openYouTubePopup(ytId, startSeconds = 0) {
        const overlay = createOverlay();
        const startParam = startSeconds > 0 ? `&start=${startSeconds}` : '';
        const iframe = document.createElement('iframe');
        iframe.src = `https://www.youtube.com/embed/${ytId}?autoplay=1&rel=0${startParam}`;
        iframe.style.width = '90vw';
        iframe.style.height = '90vh';
        iframe.style.border = 'none';
        iframe.style.borderRadius = '6px';
        iframe.style.boxShadow = '0 0 20px rgba(0,0,0,0.8)';
        iframe.allow = 'autoplay; fullscreen';
        iframe.allowFullscreen = true;

        overlay.appendChild(iframe);
        document.body.appendChild(overlay);

        overlay.addEventListener('click', () => {
            document.body.removeChild(overlay);
        });
    }

    window.addEventListener('load', observePosts);
    setInterval(observePosts, CHECK_INTERVAL);

})();
